import React, { useEffect, useState } from "react";
import AuthenticatedLayout from "@/Layouts/AuthenticatedLayout";
import { Head, Link } from "@inertiajs/inertia-react";
import __ from "@/Functions/Translate";
import { BsTagFill } from "react-icons/bs";
import { AiOutlineEye } from "react-icons/ai";
import { FcUnlock } from "react-icons/fc";
import PrimaryButton from "@/Components/PrimaryButton";
import { MdGeneratingTokens } from "react-icons/md";
import axios from "axios";
import { FaGrinStars } from "react-icons/fa";
import { Inertia } from "@inertiajs/inertia";
import { toast } from "react-toastify";
import Lightbox from "yet-another-react-lightbox";
import Thumbnails from "yet-another-react-lightbox/plugins/thumbnails";
import "yet-another-react-lightbox/styles.css";
import "yet-another-react-lightbox/plugins/thumbnails.css";

export default function SingleGallery({ inModal, gallery, closeModal }) {
  const [photos, setPhotos] = useState([]);
  const [open, setOpen] = useState(gallery.canBeViewed);
  const [lightboxPhotos, setLightboxPhotos] = useState([]);
  const thumbnailsRef = React.useRef(null);

  const fetchPhotos = async () => {
    try {
      const response = await axios.get(
        route("gallery.photos", { gallery: gallery.id })
      );

      // Assuming the response data is an array of photos
      setPhotos(response.data);

      const formattedLightboxPhotos = response.data.map((photo) => ({
        src: photo.photoUrl,
      }));

      setLightboxPhotos(formattedLightboxPhotos);
    } catch (error) {
      console.error("Error fetching photos:", error);

      if (error.response) {
        // The request was made and the server responded with a status code
        // that falls out of the range of 2xx
        if (error.response.status === 404) {
          toast.error("Gallery not found"); // Customize the error message as needed
        } else {
          toast.error(`Server Error: ${error.response.status}`);
        }
      } else if (error.request) {
        // The request was made but no response was received
        toast.error("No response from the server");
      } else {
        // Something happened in setting up the request that triggered an Error
        toast.error(`Error: ${error.message}`);
      }
    }
  };

  useEffect(() => {
    if (gallery.canBeViewed) {
      fetchPhotos();
    }
  }, [gallery.id]);

  if (gallery.canBeViewed && photos.length > 0) {
    return (
      <Lightbox
        open={open}
        close={() => {
          setOpen(false);
          closeModal();
        }}
        slides={lightboxPhotos}
        plugins={[Thumbnails]}
        thumbnails={{ ref: thumbnailsRef }}
        on={{
          click: () => {
            (thumbnailsRef.current?.visible
              ? thumbnailsRef.current?.hide
              : thumbnailsRef.current?.show)?.();
          },
        }}
      />
    );
  }

  return (
    <>
      <div className="bg-white p-3 rounded-lg dark:bg-gray-700 dark:text-white">
        <h3 className="text-lg md:text-2xl font-light text-gray-600 dark:text-white">
          {gallery.gallery_name}
        </h3>

        <div className="mt-5">
          {gallery.canBeViewed && photos.length < 1 && (
            <>{__("No photos in this gallery")}</>
          )}

          {gallery.canBeViewed && photos.length > 0 && (
            <div>
              <div className="flex flex-wrap">
                {photos.map((photo) => (
                  <img
                    onClick={(e) => setOpen(true)}
                    key={photo.id}
                    src={photo.thumbUrl}
                    alt={`Photo ${photo.id}`}
                    className="rounded-lg w-36 m-2 cursor-pointer"
                  />
                ))}
              </div>
            </div>
          )}

          {!gallery.canBeViewed && (
            <div className="flex flex-col items-center  md:flex-row space-y-5 md:space-y-0 md:space-x-5">
              <div className="relative">
                <img
                  src={gallery.previewPhotoUrl}
                  alt=""
                  className="rounded-lg w-full"
                />

                <div className="absolute top-0 left-0 text-center bg-gray-700 w-full h-full bg-opacity-25 rounded-lg ">
                  <div className="relative top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 flex flex-col justify-center">
                    <div className="w-full">
                      <div className="bg-white inline-flex bg-opacity-25 rounded-full p-2">
                        <FcUnlock className="w-12 h-12" />
                      </div>
                    </div>

                    <div>
                      <PrimaryButton
                        className="h-12 mt-5 inline-flex"
                        onClick={(e) =>
                          Inertia.visit(
                            route("gallery.unlock", {
                              gallery: gallery.id,
                            })
                          )
                        }
                      >
                        <MdGeneratingTokens className="mr-2 w-6 h-6 md:w-8 md:h-8" />
                        {__("Unlock with :tokens tokens", {
                          tokens: gallery.price,
                        })}
                      </PrimaryButton>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          )}
        </div>
      </div>
    </>
  );
}
