const { createServer } = require("http");
const { WebSocketServer } = require("ws");
const { getConfig } = require("./config.js");

const config = getConfig();
const server = createServer();

const wss = new WebSocketServer({ server });
const activeRooms = new Map();

wss.on("connection", (ws) => {
  ws.on("message", (message) => {
    try {
      const data = JSON.parse(message);
      if (data.type === "join-live-tv") {
        const roomName = `live-tv-${data.liveTvId}`;
        ws.roomName = roomName;
        if (!activeRooms.has(roomName)) {
          activeRooms.set(roomName, new Set());
        }
        activeRooms.get(roomName).add(ws);
        broadcast(roomName, {
          type: "viewer-count",
          count: activeRooms.get(roomName).size,
        });
      }

      if (data.type === "new-comment") {
        broadcast(ws.roomName, {
          type: "comment-received",
          user: data.user,
          comment: data.comment,
        });
      }
    } catch (error) {
      console.error("❌ Invalid message format", error);
    }
  });

  ws.on("close", () => {
    if (ws.roomName && activeRooms.has(ws.roomName)) {
      activeRooms.get(ws.roomName).delete(ws);
      broadcast(ws.roomName, {
        type: "viewer-count",
        count: activeRooms.get(ws.roomName).size,
      });
    }
  });
});

function broadcast(roomName, data) {
  if (activeRooms.has(roomName)) {
    activeRooms.get(roomName).forEach((client) => {
      if (client.readyState === 1) {
        client.send(JSON.stringify(data));
      }
    });
  }
}

const PORT = config.port || 8080;
server.listen(PORT, () =>
  console.log(`WebSocket server running on port ${PORT}`)
);
